
/*
     Ardunio based tube power amplifier controller.
     Mark Driedger
     v1.0    Jan 19, 2014
     (c) Mark Driedger, 2014
         
     Automatically controls power on sequencing and power off for an audio device based on sensing analog audio input signal presence
  
     When audio first appears, P0 goes high and enables power (start filament supply on tube amp).  
     After 8 seconds (filament warmup on tube amps), P1 goes high and enables the driver B+.
     After another 1 second (driver output stabilizes), P2 goes high and enables the output stage B+.
     When audio is absent for more than 120 seconds, then the audio device is turned off.
     If the manual_off input is low, then the audio device is turned off regardless of the audio input or current power on sequencing state
     If the manual_on input is low, then the audio device is turned on regardless of the audio input, going through the same sequencing of P0, P1, P2
*/

//===========[ port mapping ]======================
const int       PortExtAudio = 6;       // external audio present input (active low) from DAC or USB chip
const int       PortLoop = 7;           // use spare port to measure loop sample rate, high during loop processing
const int       PortManualOn = 8;       // manual ON override switch input, active low
const int       PortManualOff = 9;      // manual OFF override switch input, active low
const int       PortP2 = 10;            // optoisolator output port, P2, active high
const int       PortP1 = 11;            // optoisolator output port, P1, active high
const int       PortP0 = 12;            // optoisolator output port, P0, active high
const int       PortMirrorAudio = 13;   // use on board LED to mirror state of AudioPresent, active high

//===========[ global variables ]===================
int             state;                  // current state
long            CounterOn;              // counts up from 0 to DelayP0 and then DelayP1
long            CounterOff;             // counts down from DelayOff to 0
float           AudioFiltered;          // filtered audio present signal

//===========[ constants ]==========================

// Possible values of variable "state"
const int      StateOff = 0;            // all outputs off (LOW), waiting for AudioPresent or ManualOn
const int      StateP0  = 1;            // P0 output on (HIGH), waiting for CounterOn to hit DelayP0
const int      StateP1  = 2;            // P0 and P1 outputs on (HIGH), waiting for CounterOn to hit DelayP1
const int      StateP2  = 3;            // P1, P1, and P2 outputs on (HIGH), waiting for AudioPresent to become inactive

const int      Tp = 300;                // loop() processing time in usec (excluding added delayMicrosecond() call)
const int      Ts = 500;                // total desired loop() time in usec, must be > Tp
const long     loop_Fs = 1000000L/Ts;   // sample freqency in samples/sec

const long     DelayP0  =  8*loop_Fs;   // 8 second delay after P0 is active before P1 is enabled
const long     DelayP1  =  9*loop_Fs;   // 9 second delay after P0 is active before P2 is enabled (2 seconds between P1 & P2)
const long     DelayOff = 120*loop_Fs;  // 120 second delay after audio disappears before power is disabled


void setup() 
{
  state = StateOff;
  AudioFiltered = 0.0;
  
  pinMode(PortManualOn,    INPUT_PULLUP);
  pinMode(PortManualOff,   INPUT_PULLUP);
  pinMode(PortExtAudio,    INPUT_PULLUP);
  pinMode(PortP0,          OUTPUT);
  pinMode(PortP1,          OUTPUT);  
  pinMode(PortP2,          OUTPUT);
  pinMode(PortMirrorAudio, OUTPUT);
  pinMode(PortLoop,        OUTPUT); 
 }


void loop() 
{
  const int       ThresholdA = 7;         // threshold to declare audio L or R input present in ADC LSB's (relative to 1023)
  const float     ThresholdB = 0.05;      // threshold to declare AudioPresent after filtering (relative to 1.0)
  const float     alpha = 0.91;           // single pole filter constant to filter audio present signal

  boolean         AudioPresent;           // true if audio input is present
  boolean         ManualOn;               // true if manual on switch is active (low)
  boolean         ManualOff;              // true if manual off switch is active (low)
  boolean         ExtAudioPresent;        // true if external audio present pin is active (low)
  
  digitalWrite(PortLoop, HIGH);

  //read audio inputs, compare to ThresholdA and then filter 

  AudioFiltered = alpha * AudioFiltered + (1-alpha) * ((analogRead(A0) > ThresholdA) || (analogRead(A1) > ThresholdA));
  AudioPresent = AudioFiltered > ThresholdB;

  digitalWrite(PortMirrorAudio, AudioPresent);
 
  // read digital input ports
  ManualOn     = digitalRead(PortManualOn)  == LOW;
  ManualOff    = digitalRead(PortManualOff) == LOW;
  ExtAudioPresent = digitalRead(PortExtAudio) == LOW;
  
  // if manual off switch is active, then change state to StateOff
  if (ManualOff) 
  {
    state = StateOff;
    CounterOn = 0;
  }

  // implement state machine
  switch (state)
  {
    case StateOff:
      digitalWrite(PortP2,  LOW); digitalWrite(PortP1,  LOW); digitalWrite(PortP0,  LOW);
      if (ManualOn || AudioPresent || ExtAudioPresent)
      {
        state = StateP0;
        CounterOn = 0;
      }
      break;
    case StateP0:
      digitalWrite(PortP2,  LOW); digitalWrite(PortP1,  LOW); digitalWrite(PortP0, HIGH);
      if (++CounterOn > DelayP0)
        state = StateP1;
      break;
    case StateP1:
      digitalWrite(PortP2,  LOW); digitalWrite(PortP1, HIGH); digitalWrite(PortP0, HIGH);
      if (++CounterOn >= DelayP1)
      {
        state = StateP2;
        CounterOff = DelayOff;
      }
      break;
    case StateP2:
      digitalWrite(PortP2, HIGH); digitalWrite(PortP1, HIGH); digitalWrite(PortP0, HIGH);
      if (AudioPresent) 
        CounterOff = DelayOff;
      else if (--CounterOff <=0) 
        state = StateOff;
      break;
  }

  digitalWrite(PortLoop, LOW);
  delayMicroseconds(Ts-Tp);
}

